/*
THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES INC. ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE
DISCLAIMED. IN NO EVENT SHALL ANALOG DEVICES INC. BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

YOU ASSUME ANY AND ALL RISK FROM THE USE OF THIS CODE OR SUPPORT FILE.

IT IS THE RESPONSIBILITY OF THE PERSON INTEGRATING THIS CODE INTO AN APPLICATION
TO ENSURE THAT THE RESULTING APPLICATION PERFORMS AS REQUIRED AND IS SAFE.

    Module       : adf7023.c
    Description  : adf7023 interface
    Date         : 09 November 2010
    Version      : v100 09 November 2010
    Changelog    : v100 initial
*/
#include "include.h"
#include <string.h>

volatile boolean         bPacketRcv        = FALSE;
volatile boolean         bPacketTx         = FALSE;
volatile boolean         bCmdDone          = FALSE;
volatile boolean         bAESDone          = FALSE;

ADFSTA_Reg      ADF7023Status;
TyBBRAM         BBRAM;

#define DEFAULT_FREQ_DEV        50000           // 50 kHz
#define DEFAULT_DATA_RATE       200000          // 200 kbps
#define DEFAULT_CHNL_FREQ       915000000       // 915 MHz
#define FREQ_CNVRT_VAL          0.00252061538

boolean     ADF_SyncComms       (void);
void        ADF_BBRAMDefault    (TyBBRAM *pBBRAM);
void        ADF_XMit            (unsigned char ucByte,unsigned char *pData);
boolean     ADF_ReadStatus      (ADFSTA_Reg *pStatus);
boolean     ADF_WaitCmdLdr      (void);
ADF_FwState ADF_GetFwState      (void);
boolean     ADF_WaitFWState     (ADF_FwState FWState);

/*************************************************************************/
/* void ADF_FirstConnect(void)                                           */
/* Parameters:                                                           */
/*   Connecting to the ADF7023 for the first time:                       */
/*   after POR or after bottom die reset.                                */
/*   this function will be called once. BBRAM is initialised here        */
/*************************************************************************/
boolean        ADF_FirstConnect       (void)
{
   boolean        bOk = TRUE;
   ADF_FwState   FwState;

   bOk   = bOk && ADF_IssueCommandNW    (CMD_HW_RESET);

   bOk   = bOk && ADF_SyncComms   ();           // will synchronise processors, recommended seq at power up

   FwState = ADF_GetFwState();

   // handle all possible states that the ADF7023 could be in and brings it back to PHY_ON state
   while ((FwState != FW_OFF) && bOk)
       {
       switch (FwState)
           {
           case FW_BUSY:
               break;
           case FW_TX:
               bOk   = bOk && ADF_IssueCommandNW(CMD_PHY_ON);
               bOk   = bOk && ADF_WaitFWState   (FW_ON);
               break;
           case FW_RX:
               // Transition modes
               bOk   = bOk && ADF_IssueCommandNW(CMD_PHY_ON);
               bOk   = bOk && ADF_WaitFWState   (FW_ON);
               break;
           default:
               bOk   = bOk && ADF_IssueCommandNW(CMD_PHY_OFF);
               bOk   = bOk && ADF_WaitFWState   (FW_OFF);
               break;
           }
       FwState = ADF_GetFwState();
       }
   
   // Generate our default radio setup
   ADF_BBRAMDefault(&BBRAM);  

   // Update the radio
   bOk = bOk && ADF_ConfigureRadio(&BBRAM);

   // Part in the PHY_ON mode
   bOk = bOk && ADF_GoToOnState();

   return bOk;
}

/*************************************************************************/
/* void ADF_SyncComms(void)                                            */
/* Parameters:                                                           */
/*************************************************************************/
boolean        ADF_SyncComms       (void)
{
   boolean        bOk = TRUE;
   bOk   = bOk && ADF_IssueCommandNW(CMD_SYNC);    // will synchronise processors, recommended seq at power up
   bOk   = bOk && ADF_WaitCmdLdr  ();
   return bOk;
}
/*************************************************************************/
/* void ADF_SetChannelFreq(TyBBRAM *pBBRAM,unsigned long ulChannelFreq)  */
/*************************************************************************/
void ADF_SetChannelFreq(TyBBRAM *pBBRAM,unsigned long ulChannelFreq)
{
   // Use current if not otherwise specified
   if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   // The RF channel frequency bits [7:0] in Hz is set according to:
   // Frequency(Hz) = FPFD x channel_Freq[23:0] /2^16
   // where FPFD is the PFD frequency and is equal to 26MHz
   // 
   ulChannelFreq = (unsigned long)(ulChannelFreq * FREQ_CNVRT_VAL);
   pBBRAM->channel_freq_0_r                     = (ulChannelFreq >> 0) & 0xFF;
   pBBRAM->channel_freq_1_r                     = (ulChannelFreq >> 8) & 0xFF;
   pBBRAM->channel_freq_2_r                     = (ulChannelFreq >> 16)& 0xFF;
}


/*************************************************************************/
/* void ADF_SetDiscrimBW(TyBBRAM *pBBRAM,unsigned long ulDiscrimBW)      */
/*************************************************************************/
void ADF_SetDiscrimBW(TyBBRAM *pBBRAM,unsigned long ulDiscrimBW)
{
   // Use current if not otherwise specified
   if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   pBBRAM->radio_cfg_3_r = (unsigned char)ulDiscrimBW;
}

/*************************************************************************/
/* void ADF_SetDiscrimPhase(TyBBRAM *pBBRAM,unsigned long ulDiscrimPhase)      */
/*************************************************************************/
void ADF_SetDiscrimPhase(TyBBRAM *pBBRAM,unsigned long ulDiscrimPhase)
{
   // Use current if not otherwise specified
   if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   pBBRAM->radio_cfg_6_r &= ~0x03; // synth_lut_config_0 | discrim_phase[1:0] 
   pBBRAM->radio_cfg_6_r |= (ulDiscrimPhase & 0x3); // synth_lut_config_0 | discrim_phase[1:0] 
}

/*************************************************************************/
/* void ADF_SetFreqDev          (TyBBRAM *pBBRAM,unsigned long ulFreqDev)*/
/*************************************************************************/
void ADF_SetFreqDev          (TyBBRAM *pBBRAM,unsigned long ulFreqDev)
{
   // Use current if not otherwise specified
   if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   // The binary level FSK frequency deviation in Hz (defined as
   // frequency difference between carrier frequency and 1/0 tones) is
   // set according to:
   // Frequency Deviation (Hz) = Freq_Deviation[11:0] x100

   // Frequency Deviation
   pBBRAM->radio_cfg_2_r = (unsigned char)((ulFreqDev / 100) & 0xFF);
   pBBRAM->radio_cfg_1_r &= 0x0F; // Upper nibble of radio_cfg_1_r is used for Frequency Deviation[11:8]
   pBBRAM->radio_cfg_1_r |= (unsigned char)(((ulFreqDev / 100) & 0xF00) >> 4);;
}
/*************************************************************************/
/* void ADF_SetDataRate         (TyBBRAM *pBBRAM,unsigned long ulDataRate)*/
/*************************************************************************/
void ADF_SetDataRate         (TyBBRAM *pBBRAM,unsigned long ulDataRate)
{
   // Use current if not otherwise specified
   if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   // The datarate in bps is set according to:
   // DataRate (bps) = data_rate[11:0] x 100
   pBBRAM->radio_cfg_0_r = (unsigned char)((ulDataRate / 100) & 0xFF);
   pBBRAM->radio_cfg_1_r &= 0xF0; // Bottom nibble of radio_cfg_1_r is used for data_rate[11:8]
   pBBRAM->radio_cfg_1_r |= (unsigned char)(((ulDataRate / 100) & 0xF00) >> 8);;
}
/*************************************************************************/
/* void ADF_SetPALevel          (TyBBRAM *pBBRAM,unsigned long ulPALevel)*/
/*************************************************************************/
void ADF_SetPALevel          (TyBBRAM *pBBRAM,unsigned long ulPower)
{
   // Use current if not otherwise specified
   if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   // PA settings
   pBBRAM->radio_cfg_8_r = radio_cfg_8_pa_single_diff_sel_differential |
                           (ulPower << radio_cfg_8_pa_power_offset)    | 
                           radio_cfg_8_pa_ramp_16;
}


/*************************************************************************/
/* void ADF_BBRAMDefault(void)                                           */
/* Setup the default configuration parameters in all modes                */
/* Parameters:                                                           */
/*    None.                                                            */
/*************************************************************************/
void ADF_BBRAMDefault(TyBBRAM *pBBRAM)
{
   int      iOffset;
   // Initialise
   memset(pBBRAM,0x0,sizeof(TyBBRAM));

   // Configure which events will be flagged to the Cortex via interrupt
   pBBRAM->interrupt_mask_0_r                   = interrupt_mask_0_interrupt_tx_eof          | // Packet transmitted              
                                                  interrupt_mask_0_interrupt_crc_correct     | // Packet received   
                                                  interrupt_mask_0_interrupt_aes_done;
                                                    
   pBBRAM->interrupt_mask_1_r                   = 0x0;

   // These can be initialised to zero
   // Internal 16-bit count of the number of wake ups (wuc timeouts) the device has gone through 
   pBBRAM->number_of_wakeups_0_r                = 0x0;
   pBBRAM->number_of_wakeups_1_r                = 0x0;

   // This is the threshold for the number of wakeups
   // (wuc timeouts). It is a 16-bit count threshold that is compared
   // against the number_of_wakeups. When this threshold is exceeded
   // the device wakes up into the state PHY_OFF and optionally
   // generates interrupt_num_wakeups.
   pBBRAM->number_of_wakeups_irq_threshold_0_r  = 0xFF;
   pBBRAM->number_of_wakeups_irq_threshold_1_r  = 0xFF;

   // When the WUC is used and SWM is enabled), then the radio
   // powers up and enables the receiver on the channel defined in the
   // BBRAM and listens for this period of time. If no preamble pattern is
   // detected in this period, the device goes back to sleep.
   pBBRAM->rx_dwell_time_r             = 0x0;

   // Units of time used to define the rx_dwell_time time period.
   pBBRAM->parmtime_divider_r                   = 0x33;  // 995.7Hz

   // This sets the RSSI threshold when in Smart Wake Mode with RSSI
   // detection enabled.
   // Threshold (dBm) = listen_rssi_thresh - 119
   pBBRAM->swm_rssi_thresh_r                 = 0x31; // -58dBm

   // Set the channel frequency
   ADF_SetChannelFreq(pBBRAM,DEFAULT_CHNL_FREQ);

   // The datarate
   ADF_SetDataRate(pBBRAM,DEFAULT_DATA_RATE);
   
   // Frequency Deviation
   ADF_SetFreqDev          (pBBRAM,DEFAULT_FREQ_DEV);

   // RX Setting
   // Discriminator bandwidth for 50K data rate, 50k freq_dev, 200k IF freq, 10k freq error
   pBBRAM->radio_cfg_3_r = 0x31; // discriminator_bw 

  // Post-demodulator bandwidth
   pBBRAM->radio_cfg_4_r = (unsigned char)((0.75*(DEFAULT_DATA_RATE/1000))/2); // post_demod_bw

   // Reserved
   pBBRAM->radio_cfg_5_r = 0x00; // reserved

   // radio_cfg_6
   pBBRAM->radio_cfg_6_r = 0x02; // synth_lut_config_0 | discrim_phase[1:0] 

   // agc_mode, Synth_Lut_control, Synth_LUT_config_1 
   pBBRAM->radio_cfg_7_r =  SET_BITS(0,
                                     radio_cfg_7_synth_lut_config_1_numbits,
                                     radio_cfg_7_synth_lut_config_1_offset,
                                     0)                                                |
                                     radio_cfg_7_synth_lut_control_predef_rx_predef_tx |
                                     radio_cfg_7_agc_lock_mode_lock_after_preamble     ;

   // PA settings
   pBBRAM->radio_cfg_8_r = radio_cfg_8_pa_single_diff_sel_differential |
                           radio_cfg_8_pa_power_setting_63             | 
                           radio_cfg_8_pa_ramp_16;

   // Modulation/Demodulation Scheme
   // FSK/FSK/150 Hz receiver filter bandwidth
   pBBRAM->radio_cfg_9_r = radio_cfg_9_demod_scheme_FSK        |
                           radio_cfg_9_mod_scheme_2_level_FSK  |
                           radio_cfg_9_ifbw_200kHz;
   // AFC
   pBBRAM->radio_cfg_10_r = radio_cfg_10_afc_polarity_fixed_value            |  
                            radio_cfg_10_afc_scheme_fixed_value              |
                            radio_cfg_10_afc_lock_mode_lock_after_preamble;

   // Sets the AFC PI controller proportional gain. 
   // Sets the AFC PI controller integral gain. 
   pBBRAM->radio_cfg_11_r   = radio_cfg_11_afc_kp_2_power_3 | radio_cfg_11_afc_ki_2_power_7; 

   // image_reject_cal_phase
   pBBRAM->image_reject_cal_phase_r   = 0x0;
   // image_reject_cal_amplitude
   pBBRAM->image_reject_cal_amplitude_r   = 0x0; 

   pBBRAM->mode_control_r = mode_control_swm_en_disabled                        |   
                            mode_control_bb_cal_enabled                         |   
                            mode_control_swm_rssi_qual_disabled                 |   
                            mode_control_tx_auto_turnaround_disabled            |   
                            mode_control_rx_auto_turnaround_disabled            |   
                            mode_control_custom_trx_synth_lock_time_en_disabled |   
                            mode_control_ext_lna_en_disabled                    |   
                            mode_control_ext_pa_en_disabled                     ;   

   // Number of preamble bit errors in 24 bit window (RX)
   // value of <4 bit errors is recommended to prevent false preamble detections
   pBBRAM->preamble_match_r = preamble_match_0_in_24_win;

   // Symbol Paramters
   pBBRAM->symbol_mode_r = symbol_mode_symbol_length_8_bit          |
                           symbol_mode_data_whitening_disabled      |
                           symbol_mode_eight_ten_enc_disabled       |
                           symbol_mode_prog_crc_en_disabled         | // Default CRC selected (x16 + x15 + x2 + 1)
                           symbol_mode_manchester_enc_enabled ;

   // Length of (TX) preamble in bytes. Example a value of decimal 3 
   // results in a preamble of 24 bits.
   pBBRAM->preamble_len_r = 0x20;  

   // crc_poly[15:0], which sets the CRC polynomial (using default in symbol_mode). 
   pBBRAM->crc_poly_0_r  = 0x00;
   pBBRAM->crc_poly_1_r  = 0x00;

   // Sets the sync word error tolerance in bits. 
   // Sets the sync word length in bits. 24 bits is the maximum. 
   // Note that the sync word matching length can be any value up to 24 bits
   // , but the transmitted sync word pattern is a multiple of 8 bits. 
   // Hence, for non-byte-length sync words, the transmitted sync pattern 
   // should be filled out with the preamble pattern.
   pBBRAM->sync_control_r = sync_control_sync_error_tol_0_errors_allowed | 
                            sync_control_sync_word_length_8; 
   // The sync word pattern is transmitted most significant bit first 
   // starting with sync_byte[7:0].
   // For non-byte length sync words the reminder of the least 
   // significant byte should be stuffed with preamble.
   // If sync_word_length length is >16 bits then sync_byte_0, 
   // sync_byte_1 and sync_byte_2 are all transmitted for a total of 24 bits. 
   // If sync_word_length is between 8 and 15 then sync_byte_1 and sync_byte_2 
   // are transmitted. 
   // If sync_word_length is between 1 and 7 then sync_byte_2 is 
   // transmitted for a total of 8 bits.
   // If the sync word length is 0 then no sync bytes are transmitted.
   pBBRAM->sync_byte_0_r  = 0x0; 
   pBBRAM->sync_byte_1_r  = 0x0;
   pBBRAM->sync_byte_2_r  = 0xA7;


   // Address in Packet RAM of transmit packet. This address 
   // indicates to the comms processor the location of the 
   // first byte of the transmit packet
   pBBRAM->tx_base_adr_r            = PKT_RAM_BASE_PTR; 

   // Address in Packet RAM of receive packet. The communications 
   // processor will write any qualified received packet to Packet RAM, 
   // starting at this memory location.
   pBBRAM->rx_base_adr_r            = PKT_RAM_BASE_PTR;    
                                                         
   // Various packet options
   pBBRAM->packet_length_control_r  = packet_length_control_data_byte_lsb         | // LSB
                                      packet_length_control_packet_len_variable   | // Variable packet length
                                      packet_length_control_crc_en_yes            | // CRC Enabled
                                      packet_length_control_data_mode_packet      | // No sport
                                      packet_length_control_length_offset_minus0;   // For variable length packets where the first byte (length) needs to be adjusted

   pBBRAM->packet_length_max_r      = PKT_MAX_PKT_LEN;

   // Set to 0x00
   pBBRAM->static_reg_fix_r         = 0x0;

   // Location of first byte of address information in packet RAM (relative to rx_base)
   pBBRAM->address_match_offset_r   = ADR_MATCH_OFFSET;

   iOffset = 0x0;
   // Number of bytes in address field (NADR)
   pBBRAM->address_filtering_r[iOffset++]     = 0x2; // Using a two byte address (0 for addr matching disabled)
   // Two byte addess - 0x01FF
   pBBRAM->address_filtering_r[iOffset++]     = 0x01;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0xFF;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0xFF;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0xFF;

   // Number of bytes in address field (NADR)
   pBBRAM->address_filtering_r[iOffset++]     = 0x2; // Using a two byte address (0 for addr matching disabled)
   // Two byte addess - 0x0F0F
   pBBRAM->address_filtering_r[iOffset++]     = 0x0F;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0xFF;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0x0F;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0xFF;

   // Number of bytes in address field (NADR)
   pBBRAM->address_filtering_r[iOffset++]     = 0x2; // Using a two byte address (0 for addr matching disabled)
   // Two byte addess - 0x5061
   pBBRAM->address_filtering_r[iOffset++]     = 0x50;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0xFF;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0x61;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0xFF;                 

   // Number of bytes in address field (NADR)
   pBBRAM->address_filtering_r[iOffset++]     = 0x0;                 

   // 
   pBBRAM->address_filtering_r[iOffset++]     = 0x0;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0x0;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0x0;                 
   pBBRAM->address_filtering_r[iOffset++]     = 0x0;                 


   pBBRAM->rx_synth_lock_time_r       = 0x0; 
   pBBRAM->rx_synth_lock_time_r       = 0x0; 
}

/*************************************************************************/
/* void ADF_ConfigureRadio(void)                                         */
/* Parameters:                                                           */
/*************************************************************************/
boolean        ADF_ConfigureRadio       (TyBBRAM *pBBRAM)
{
   boolean        bOk = TRUE;

   // Use current if not otherwise specified
   if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   // Commit the local BBRAM to the Radio
   bOk = bOk && ADF_MMapWrite(BBRam_MMemmap_Start, sizeof(TyBBRAM), (unsigned char *)pBBRAM);

   // Part in the PHY_ON mode
   bOk = bOk && ADF_GoToOnState();

   // Update MCRs
   bOk = bOk && ADF_IssueCommand(CMD_CONFIG_DEV);

   return bOk;
}


/*************************************************************************/
/* void ADF_BBRAMReadBack(void)                                          */
/* Parameters:                                                           */
/*    Connecting to the ADF7023 for the first time.                      */
/*    BBRAM initialisation.                                              */
/*************************************************************************/
boolean ADF_BBRAMReadBack       (void)
{
   boolean        bOk = TRUE;
   bOk = bOk && ADF_MMapRead(BBRam_MMemmap_Start, sizeof(TyBBRAM), (unsigned char *)&BBRAM);
   return bOk;
}


/*************************************************************************/
/* void ADF_GoToOnState(void)                    */
/* Parameters:          */
/*************************************************************************/
boolean ADF_GoToOnState(void)
{
   boolean        bOk = TRUE;
   bOk   = bOk && ADF_IssueCommand(CMD_PHY_ON); // Enter PHY_ON mode
   bOk   = bOk && ADF_WaitFWState     (FW_ON);
   return bOk;
}

/*************************************************************************/
/* void ADF_ReceivePacket(void)                                        */
/* Parameters:                                                           */
/*************************************************************************/
boolean        ADF_ReceivePacket       (void)
{
   boolean        bOk = TRUE;

   bOk   = bOk && ADF_IssueCommand    (CMD_PHY_ON);
   bOk   = bOk && ADF_WaitFWState     (FW_ON);

   bOk   = bOk && ADF_IssueCommand    (CMD_PHY_RX);

   // Radio will stay on until a packet is received. Interrupt sets a flag
   return bOk;
}

// Packet Definition (Total 240 bytes)
// Byte 0  : Length
// Byte 1  : Addr1
// Byte 2  : Addr2
// Byte 3  : Variable Length Payload Data (237 max)
// ....
// Byte 239: Variable Length Payload Data
/*************************************************************************/
/* void ADF_TransmitPacket(void)                                        */
/* Parameters:                                                           */
/*************************************************************************/
boolean        ADF_TransmitPacket       (U8 * pucTXData,U8 ucLen)
{
   boolean        bOk       = TRUE;
   unsigned char  ParamTX   = PARAM_TX_NORMAL_PACKET;
   unsigned char  ucHdr      [0x3];

   if ((ucLen + sizeof(ucHdr)) <= PKT_MAX_PKT_LEN)
       {
       // Packet Header 
       ucHdr[LEN_OFFSET]           = ucLen + sizeof(ucHdr);
       ucHdr[ADR_MATCH_OFFSET]     = 0x01;
       ucHdr[ADR_MATCH_OFFSET+0x1] = 0xFF;
       bOk   = bOk && ADF_MMapWrite     (PKT_RAM_BASE_PTR,sizeof(ucHdr),ucHdr);
       // Packet Payload 
       bOk   = bOk && ADF_MMapWrite     (PKT_RAM_BASE_PTR+sizeof(ucHdr),ucLen,pucTXData);

       // This would have been changed if infinite premable or carrier was sent.
       bOk   = bOk && ADF_MMapWrite     (PR_var_tx_mode_ADR,sizeof(ParamTX),&ParamTX);

       // Transmit the packet
       bOk   = bOk && ADF_IssueCommand    (CMD_PHY_ON);
       bOk   = bOk && ADF_WaitFWState     (FW_ON);
       bOk   = bOk && ADF_IssueCommand    (CMD_PHY_TX);
       // We will get an interrupt when the transmission is actually complete
       }
   else
       bOk = FALSE;

   return bOk;
}

/*************************************************************************/
/* void ADF_TransmitCarrier(void)                                        */
/* Parameters:                                                           */
/*************************************************************************/
boolean ADF_TransmitCarrier(void)

{
   boolean        bOk       = TRUE;
   unsigned char  ParamTX   = PARAM_TX_CARRIER_FOREVER;

   // Test mode needs to be set, before entry to PHY_TX
   bOk   = bOk && ADF_MMapWrite     (PR_var_tx_mode_ADR,sizeof(ParamTX),&ParamTX);
   bOk   = bOk && ADF_IssueCommand  (CMD_PHY_ON);
   bOk   = bOk && ADF_WaitFWState   (FW_ON);
   bOk   = bOk && ADF_IssueCommand  (CMD_PHY_TX);

   return bOk;
}

/*************************************************************************/
/* void ADF_TransmitPreamble(void)                                        */
/* Parameters:                                                           */
/*************************************************************************/
boolean ADF_TransmitPreamble(void)

{
   boolean        bOk       = TRUE;
   unsigned char  ParamTX   = PARAM_TX_PREAMBLE_FOREVER;

   // Test mode needs to be set, before entry to PHY_TX
   bOk   = bOk && ADF_MMapWrite     (PR_var_tx_mode_ADR,sizeof(ParamTX),&ParamTX);
   bOk   = bOk && ADF_IssueCommand  (CMD_PHY_ON);
   bOk   = bOk && ADF_WaitFWState   (FW_ON);
   bOk   = bOk && ADF_IssueCommand  (CMD_PHY_TX);

   return bOk;
}

/******************************************************************************/
/* Function    : ADF_XMitByte                                                 */
/* Description : Transmit a byte over SPI and wait for response               */
/*               Update status or return data                                 */
/******************************************************************************/
void ADF_XMit(unsigned char ucByte,unsigned char *pData)
{
   SEND_SPI(ucByte);  // Send byte
   WAIT_SPI_RX;  // wait for data received status bit
   if(pData)
      *pData = READ_SPI;
   else
      (void)READ_SPI;
}

/******************************************************************************/
/* Function    : MMapReadByte                                                 */
/* Description : Read Byte from specified memory map address                  */
/******************************************************************************/
unsigned char ADF_MMapRead    (unsigned long ulAdr, unsigned long ulLen, unsigned char *pData)
{
   int iTmp,RetVal = TRUE;
   int i = 0x0;
   if(RetVal)
      {
      ADFSTA_MemCmd Cmd;
      // Assemble the command
      Cmd.Bits.Cmd  = SPI_MEM_RD >> 3;
      Cmd.Bits.AdrH = (ulAdr & 0x700) >> 8;
      Cmd.Bits.AdrL = ulAdr & 0xFF;
      // De-assert SPI chip select
      ADF_CSN_DEASSERT;
      // Assert SPI chip select
      ADF_CSN_ASSERT;

      // Wait for MISO line to go high
      // Wait for ADF to wake up (only necessary for first byte in packet)
      iTmp = 0; // clear before following line
      // monitor MISO line 
      while (0 == iTmp && i < 1000) { 
          iTmp = ADF_MISO_IN; // bits 7:0 are current POrt1 data input
          i++;
      }
      if (1000 == i)
         RetVal = FALSE; //
      else
         {
         // Send first byte (SPI_MEMR_RD + Bytes)
         ADF_XMit(Cmd.ucBytes[0x0],NULL);
         // Send Second byte remainder of addrress
         ADF_XMit(Cmd.ucBytes[0x1],NULL);
         // Send NOP
         ADF_XMit(SPI_NOP,NULL);
         // Send NOP Data available now
         while(ulLen--)
            ADF_XMit(SPI_NOP,pData++);
         }
      ADF_CSN_DEASSERT; // De-assert SPI chip select
      }
   return RetVal;
}
/******************************************************************************/
/* Function    : MMapWrite                                                    */
/* Description : Write Byte to specified memory map address                   */
/******************************************************************************/
unsigned char ADF_MMapWrite(unsigned long ulAdr,
                            unsigned long ulLen,
                            unsigned char *pData)
{
   int iTmp,RetVal = TRUE;
   int i = 0x0;

   if(RetVal)
      {
      ADFSTA_MemCmd Cmd;

      // Assemble the command
      Cmd.Bits.Cmd  = SPI_MEM_WR >> 3;
      Cmd.Bits.AdrH = (ulAdr & 0x700) >> 8;
      Cmd.Bits.AdrL = ulAdr & 0xFF;

      // De-assert SPI chip select
      ADF_CSN_DEASSERT;
      // Assert SPI chip select
      ADF_CSN_ASSERT;

      // Wait for MISO line to go high
      // Wait for ADF to wake up (only necessary for first byte in packet)
      //
      iTmp = 0; // clear before following line
      // monitor MISO line (P1.5)
      while (0 == iTmp && i < 1000) { //check whether bit 5 is high
          iTmp = ADF_MISO_IN; // bits 7:0 are current POrt1 data input
          i++;
      }
      if (1000 == i)
         RetVal = FALSE; //
      else
         {
         // Send first byte (SPI_MEMR_WR + Bytes)
         ADF_XMit(Cmd.ucBytes[0x0],NULL);
         // Send Second byte remainder of addrress
         ADF_XMit(Cmd.ucBytes[0x1],NULL);
         // Send Data
         //
         while(ulLen--)
            ADF_XMit(*(pData++),NULL);
         }
      // De-assert SPI chip select
      ADF_CSN_DEASSERT;
      }

   return RetVal;
}

/******************************************************************************/
/* Function    : ADF_IssueCommandNW                                           */
/* Description : Issue specified command                                      */
/******************************************************************************/
boolean  ADF_IssueCommandNW(unsigned char Cmd)
{
   int      iTmp;
   boolean  RetVal = TRUE;
   int      i = 0x0;

   // De-assert SPI chip select
   ADF_CSN_DEASSERT;
   // Assert SPI chip select
   ADF_CSN_ASSERT;

   // Wait for MISO line to go high
   // Wait for ADF to wake up (only necessary for first byte in packet)
   iTmp = 0; // clear before following line
   // monitor MISO line (P1.5)
   while (0 == iTmp && i < 1000) { //check whether bit 5 is high
       iTmp = ADF_MISO_IN; // bits 7:0 are current POrt1 data input
       i++;
   }
   if (1000 == i)
      RetVal = FALSE;
   else
      {
      ADF_XMit(Cmd,NULL); // Send Command
      }
   ADF_CSN_DEASSERT;   // De-assert SPI chip select
   return RetVal;
}
/******************************************************************************/
/* Function    : ADF_IssueCommand                                             */
/* Description : Issue specified command                                      */
/******************************************************************************/
boolean  ADF_IssueCommand(unsigned char Cmd)
{
   boolean  RetVal = TRUE;
   RetVal = RetVal && ADF_WaitCmdLdr();
   if(RetVal)
      RetVal = ADF_IssueCommandNW(Cmd);
   return RetVal;
}


/*! 
    \fn      boolean  ADF_ReadStatus(ADFSTA_Reg *pStatus)
    \brief   Read the SPI Status Byte.

             This should be done before any other functions are called.

             A wire reset, followed by the switch sequence from JTAG to SW
             , followed again by a wire reset is performed. This is suitable
             to put either a SW-DP or SWJ-DP into IDLE mode in Serial Wire 
             mode.

             The device that is present is detected and the Watchdog is
             turned off if it is not already locked on.


    \param   pStatus      Pointer to storage. 

    \return  boolean TRUE, if function successful, else FALSE
*/
boolean  ADF_ReadStatus(ADFSTA_Reg *pStatus)
{
   int      iTmp;
   boolean  RetVal = TRUE;
   int      i = 0x0;

   ADF_CSN_DEASSERT; // De-assert SPI chip select
   ADF_CSN_ASSERT;   // Assert SPI chip select

   // Wait for MISO line to go high after assertion of CSN
   iTmp = 0; 

   // monitor MISO line (P1.5)
   while (0 == iTmp && i < 1000) 
       {
       iTmp = ADF_MISO_IN; 
       i++;
       }

   if (1000 == i)
      RetVal = FALSE; // ADF7023 didn't respond
   else
      {
      // Send Command
      ADF_XMit(SPI_NOP,NULL);
      ADF_XMit(SPI_NOP,(unsigned char *)pStatus);
      }
   ADF_CSN_DEASSERT; // De-assert SPI chip select
   return RetVal;
}

/******************************************************************************/
/* Function    : ADF_WaitCmdLdr                                               */
/* Description : Wait for the command loader to be empty                      */
/******************************************************************************/
boolean  ADF_WaitCmdLdr(void)
{
   boolean  RetVal = TRUE;
   do
      {
      RetVal   = ADF_ReadStatus(&ADF7023Status);
      }
   while(RetVal && ((ADF7023Status.Bits.cmd_loader_empty == 0x0)));
   return RetVal;
}

/******************************************************************************/
/* Function    : ADF_GetFwState                                                */
/* Description :                          */
/******************************************************************************/
ADF_FwState ADF_GetFwState(void)
{
  (void)ADF_ReadStatus(&ADF7023Status);
  return ADF7023Status.Bits.fw_state;
}


/******************************************************************************/
/* Function    : ADF_WaitFWState                                                */
/* Description : Wait until firmware state is reached                          */
/******************************************************************************/
boolean  ADF_WaitFWState(ADF_FwState FWState)
{
   boolean  RetVal = TRUE;
   do
      {
      RetVal   = ADF_ReadStatus(&ADF7023Status);
      }
   while(RetVal && (ADF7023Status.Bits.fw_state != FWState));
   return RetVal;
}


//******************************************************************************
// Function    : Ext_Int8_Handler
// Description :
// *****************************************************************************
void ADF_CheckInt (void)
{
   boolean  bOk = TRUE;

   unsigned char ucInt0;
   unsigned char ucInt1;

   bOk = bOk && ADF_MMapRead(MCR_interrupt_source_0_Adr, 0x1, (unsigned char *)&ucInt0);
   bOk = bOk && ADF_MMapRead(MCR_interrupt_source_1_Adr , 0x1, (unsigned char *)&ucInt1);

   if (ucInt0 & interrupt_mask_0_interrupt_premable_detect)
       {

       }
   if (ucInt0 & interrupt_mask_0_interrupt_sync_detect)
       {

       }
   if (ucInt0 & interrupt_mask_0_interrupt_crc_correct)
       {
       bPacketRcv = TRUE;
       }
   if (ucInt0 & interrupt_mask_0_interrupt_address_match)
       {
	   // This is the valid packet indicator!!!"
       }
   if (ucInt0 & interrupt_mask_0_interrupt_tx_eof)
       {
       bPacketTx = TRUE;
       }
   if (ucInt0 & interrupt_mask_0_interrupt_aes_done)
       {
       bAESDone = TRUE;
       }
   if (ucInt0 & interrupt_mask_0_interrupt_swm_rssi_det)
       {

       }
   if (ucInt0 & interrupt_mask_0_interrupt_num_wakeups)
       {
       }

   if (ucInt1 & interrupt_mask_1_cmd_finished)
       {
       bCmdDone = TRUE;
       }

   // Clear all the interrupts that we have just handleed
   bOk = bOk && ADF_MMapWrite(MCR_interrupt_source_0_Adr, 0x1, (unsigned char *)&ucInt0);
   bOk = bOk && ADF_MMapWrite(MCR_interrupt_source_1_Adr , 0x1, (unsigned char *)&ucInt1);
}

//******************************************************************************
// Function    : Ext_Int8_Handler
// Description :
// *****************************************************************************
void Ext_Int8_Handler (void)
{
    ADF_CheckInt();
    // Clear the interrupt
    EICLR = EICLR_IRQ8;
}

/******************************************************************************/
/* void ADF_PrintBBRAM(TyBBRAM *pBBRAM)                                                      */
/* Parameters: pBBRAM                                                           */
/******************************************************************************/
void ADF_PrintBBRAM(TyBBRAM *pBBRAM)
{
  unsigned long   ulTmp;
  int             iOffset;
  unsigned char * pTmp;
  int             iCnt;


  // USe current if not otherwise specified
  if (pBBRAM == NULL)
      pBBRAM = &BBRAM;

   printf("\n*************\nDUMPING BBRAM\n*************\n");

   // MAC Interrups
   printf("interrupt_mask_0_r = 0x%02X ",pBBRAM->interrupt_mask_0_r);
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_num_wakeups)
       printf("\n   interrupt_mask_0_interrupt_num_wakeups");
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_swm_rssi_det)
       printf("\n   interrupt_mask_0_interrupt_swm_rssi_det");
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_aes_done)
       printf("\n   interrupt_mask_0_interrupt_aes_done");
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_tx_eof)
       printf("\n   interrupt_mask_0_interrupt_tx_eof");
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_address_match)
       printf("\n   interrupt_mask_0_interrupt_address_match");
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_crc_correct)
       printf("\n   interrupt_mask_0_interrupt_crc_correct");
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_sync_detect)
       printf("\n   interrupt_mask_0_sync_detect");
   if(pBBRAM->interrupt_mask_0_r & interrupt_mask_0_interrupt_premable_detect)
       printf("\n   interrupt_mask_0_interrupt_premable_detect");
   printf("\n");

   // PHY Interrupts
   printf("interrupt_mask_1_r = 0x%02X\n",pBBRAM->interrupt_mask_1_r);
   if(pBBRAM->interrupt_mask_1_r & interrupt_mask_1_battery_alarm)
       printf("\n   interrupt_mask_1_battery_alarm ");
   if(pBBRAM->interrupt_mask_1_r & interrupt_mask_1_cmd_ready     )
       printf("\n   interrupt_mask_1_cmd_ready      ");
   if(pBBRAM->interrupt_mask_1_r & interrupt_mask_1_wuc_timeout  )
       printf("\n   interrupt_mask_1_wuc_timeout   ");
   if(pBBRAM->interrupt_mask_1_r & interrupt_mask_1_spi_ready    )
       printf("\n   interrupt_mask_1_spi_ready     ");
   if(pBBRAM->interrupt_mask_1_r & interrupt_mask_1_cmd_finished     )
       printf("\n   interrupt_mask_1_cmd_finished      ");
   printf("\n");


   printf("number_of_wakeups_0_r = 0x%02X\n",pBBRAM->number_of_wakeups_0_r);
   printf("\n");

   printf("number_of_wakeups_1_r = 0x%02X\n",pBBRAM->number_of_wakeups_1_r);
   printf("\n");

   printf("number_of_wakeups_irq_threshold_0_r = 0x%02X\n",pBBRAM->number_of_wakeups_irq_threshold_0_r);
   printf("\n");

   printf("number_of_wakeups_irq_threshold_1_r = 0x%02X\n",pBBRAM->number_of_wakeups_irq_threshold_1_r);
   printf("\n");

   printf("rx_dwell_time_r = 0x%02X\n",pBBRAM->rx_dwell_time_r);
   printf("\n");

   printf("parmtime_divider_r = 0x%02X\n",pBBRAM->parmtime_divider_r);
   printf("\n");

   printf("swm_rssi_thresh_r = 0x%02X\n",pBBRAM->swm_rssi_thresh_r);
   printf("\n");

   printf("\nchannel_freq_0_r = 0x%02X",pBBRAM->channel_freq_0_r);
   printf("\nchannel_freq_1_r = 0x%02X",pBBRAM->channel_freq_1_r);
   printf("\nchannel_freq_2_r = 0x%02X",pBBRAM->channel_freq_2_r);
   ulTmp = MAKE_ULONG(pBBRAM->channel_freq_0_r,pBBRAM->channel_freq_1_r,pBBRAM->channel_freq_2_r,0);
   printf("\n   Channel Frequency  %ld Hz",(unsigned long)(ulTmp / FREQ_CNVRT_VAL ));
   printf("\n");

   printf("\nradio_cfg_0_r = 0x%02X",pBBRAM->radio_cfg_0_r);
   printf("\nradio_cfg_1_r = 0x%02X",pBBRAM->radio_cfg_1_r);
   printf("\nradio_cfg_2_r = 0x%02X",pBBRAM->radio_cfg_2_r);
   ulTmp = MAKE_ULONG(pBBRAM->radio_cfg_0_r,pBBRAM->radio_cfg_1_r & 0x0F,0,0);
   printf("\n   Data Rate  %ld Hz",ulTmp * 100);
   ulTmp = MAKE_ULONG(pBBRAM->radio_cfg_2_r,(pBBRAM->radio_cfg_1_r & 0xF0) >> 4,0,0);
   printf("\n   Freq Dev   %ld Hz",ulTmp * 100);
   printf("\n");

   printf("\nradio_cfg_3_r = 0x%02X",pBBRAM->radio_cfg_3_r);
   printf("\n");

   printf("\nradio_cfg_4_r = 0x%02X",pBBRAM->radio_cfg_4_r);
   printf("\n");

   printf("\nradio_cfg_5_r = 0x%02X",pBBRAM->radio_cfg_5_r);
   printf("\n");

   printf("\nradio_cfg_6_r = 0x%02X",pBBRAM->radio_cfg_6_r);
   printf("\n");

   printf("\nradio_cfg_7_r = 0x%02X",pBBRAM->radio_cfg_7_r);
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_7_r,radio_cfg_7_agc_lock_mode_numbits,radio_cfg_7_agc_lock_mode_offset);
   switch (ulTmp)
       {
       case radio_cfg_7_agc_lock_mode_free_running       :
           printf("\n   radio_cfg_7_agc_lock_mode_free_running       ");
           break;
       case radio_cfg_7_agc_lock_mode_manual             :
           printf("\n   radio_cfg_7_agc_lock_mode_manual             ");
           break;
       case radio_cfg_7_agc_lock_mode_hold               :
           printf("\n   radio_cfg_7_agc_lock_mode_hold               ");
           break;
       case radio_cfg_7_agc_lock_mode_lock_after_preamble:
           printf("\n   radio_cfg_7_agc_lock_mode_lock_after_preamble");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_7_r,radio_cfg_7_synth_lut_control_numbits,radio_cfg_7_synth_lut_control_offset);
   switch (ulTmp)
       {
       case radio_cfg_7_synth_lut_control_predef_rx_predef_tx:
           printf("\n   radio_cfg_7_synth_lut_control_predef_rx_predef_tx");
           break;
       case radio_cfg_7_synth_lut_control_custom_rx_predef_tx:
           printf("\n   radio_cfg_7_synth_lut_control_custom_rx_predef_tx");
           break;
       case radio_cfg_7_synth_lut_control_predef_rx_custom_tx:
           printf("\n   radio_cfg_7_synth_lut_control_predef_rx_custom_tx");
           break;
       case radio_cfg_7_synth_lut_control_custom_rx_custom_tx:
           printf("\n   radio_cfg_7_synth_lut_control_custom_rx_custom_tx");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_7_r,radio_cfg_7_synth_lut_config_1_numbits,radio_cfg_7_synth_lut_config_1_offset);
   printf("\n   radio_cfg_7_synth_lut_config_1 0x%lX",ulTmp);
   printf("\n");

   printf("\nradio_cfg_8_r = 0x%02X",pBBRAM->radio_cfg_8_r);

   if(pBBRAM->radio_cfg_8_r & radio_cfg_8_pa_single_diff_sel_differential)
       printf("\n   radio_cfg_8_pa_single_diff_sel_differential");
   else
       printf("\n   radio_cfg_8_pa_single_diff_sel_single_ended");


   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_8_r,radio_cfg_8_pa_ramp_numbits,radio_cfg_8_pa_ramp_offset);
   switch (ulTmp)
       {
       case radio_cfg_8_pa_ramp_256     :
           printf("\n   radio_cfg_8_pa_ramp_256     ");
           break;
       case radio_cfg_8_pa_ramp_128     :
           printf("\n   radio_cfg_8_pa_ramp_128     ");
           break;
       case radio_cfg_8_pa_ramp_64      :
           printf("\n   radio_cfg_8_pa_ramp_64      ");
           break;
       case radio_cfg_8_pa_ramp_32      :
           printf("\n   radio_cfg_8_pa_ramp_32      ");
           break;
       case radio_cfg_8_pa_ramp_16      :
           printf("\n   radio_cfg_8_pa_ramp_16      ");
           break;
       case radio_cfg_8_pa_ramp_8       :
           printf("\n   radio_cfg_8_pa_ramp_8       ");
           break;
       case radio_cfg_8_pa_ramp_4       :
           printf("\n   radio_cfg_8_pa_ramp_4       ");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_8_r,radio_cfg_8_pa_power_numbits,radio_cfg_8_pa_power_offset);
   switch (ulTmp)
       {
       case radio_cfg_8_pa_power_setting_3 :
           printf("\n   radio_cfg_8_pa_power_setting_3 ");
           break;
       case radio_cfg_8_pa_power_setting_7 :
           printf("\n   radio_cfg_8_pa_power_setting_7 ");
           break;
       case radio_cfg_8_pa_power_setting_11:
           printf("\n   radio_cfg_8_pa_power_setting_11");
           break;
       case radio_cfg_8_pa_power_setting_15:
           printf("\n   radio_cfg_8_pa_power_setting_15");
           break;
       case radio_cfg_8_pa_power_setting_19:
           printf("\n   radio_cfg_8_pa_power_setting_19");
           break;
       case radio_cfg_8_pa_power_setting_23:
           printf("\n   radio_cfg_8_pa_power_setting_23");
           break;
       case radio_cfg_8_pa_power_setting_27:
           printf("\n   radio_cfg_8_pa_power_setting_27");
           break;
       case radio_cfg_8_pa_power_setting_31:
           printf("\n   radio_cfg_8_pa_power_setting_31");
           break;
       case radio_cfg_8_pa_power_setting_35:
           printf("\n   radio_cfg_8_pa_power_setting_35");
           break;
       case radio_cfg_8_pa_power_setting_39:
           printf("\n   radio_cfg_8_pa_power_setting_39");
           break;
       case radio_cfg_8_pa_power_setting_43:
           printf("\n   radio_cfg_8_pa_power_setting_43");
           break;
       case radio_cfg_8_pa_power_setting_47:
           printf("\n   radio_cfg_8_pa_power_setting_47");
           break;
       case radio_cfg_8_pa_power_setting_51:
           printf("\n   radio_cfg_8_pa_power_setting_51");
           break;
       case radio_cfg_8_pa_power_setting_55:
           printf("\n   radio_cfg_8_pa_power_setting_55");
           break;
       case radio_cfg_8_pa_power_setting_59:
           printf("\n   radio_cfg_8_pa_power_setting_59");
           break;
       case radio_cfg_8_pa_power_setting_63:
           printf("\n   radio_cfg_8_pa_power_setting_63");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }


   printf("\n");

   printf("\nradio_cfg_9_r = 0x%02X",pBBRAM->radio_cfg_9_r);
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_9_r,radio_cfg_9_ifbw_numbits,radio_cfg_9_ifbw_offset);
   switch (ulTmp)
       {
       case radio_cfg_9_ifbw_100kHz:
           printf("\n   radio_cfg_9_ifbw_100kHz");
           break;
       case radio_cfg_9_ifbw_150kHz:
           printf("\n   radio_cfg_9_ifbw_150kHz");
           break;
       case radio_cfg_9_ifbw_200kHz:
           printf("\n   radio_cfg_9_ifbw_200kHz");
           break;
       case radio_cfg_9_ifbw_300kHz:
           printf("\n   radio_cfg_9_ifbw_300kHz");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_9_r,radio_cfg_9_mod_scheme_numbits,radio_cfg_9_mod_scheme_offset);
   switch (ulTmp)
       {
       case radio_cfg_9_mod_scheme_2_level_FSK :
           printf("\n   radio_cfg_9_mod_scheme_2_level_FSK ");
           break;
       case radio_cfg_9_mod_scheme_2_level_GFSK:
           printf("\n   radio_cfg_9_mod_scheme_2_level_GFSK");
           break;
       case radio_cfg_9_mod_scheme_OOK         :
           printf("\n   radio_cfg_9_mod_scheme_OOK         ");
           break;
       case radio_cfg_9_mod_scheme_carrier_only:
           printf("\n   radio_cfg_9_mod_scheme_carrier_only");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_9_r,radio_cfg_9_demod_scheme_numbits,radio_cfg_9_demod_scheme_offset);
   switch (ulTmp)
       {
       case radio_cfg_9_demod_scheme_FSK :
           printf("\n   radio_cfg_9_demod_scheme_FSK ");
           break;
       case radio_cfg_9_demod_scheme_OOK :
           printf("\n   radio_cfg_9_demod_scheme_OOK ");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   printf("\n");

   printf("\nradio_cfg_10_r = 0x%02X",pBBRAM->radio_cfg_10_r);
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_10_r,radio_cfg_10_afc_polarity_numbits,radio_cfg_10_afc_polarity_offset);
   switch (ulTmp)
       {
       case radio_cfg_10_afc_polarity_fixed_value:
           printf("\n   radio_cfg_10_afc_polarity_fixed_value");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_10_r,radio_cfg_10_afc_scheme_numbits,radio_cfg_10_afc_scheme_offset);
   switch (ulTmp)
       {
       case radio_cfg_10_afc_scheme_fixed_value:
           printf("\n   radio_cfg_10_afc_scheme_fixed_value");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_10_r,radio_cfg_10_afc_lock_mode_numbits,radio_cfg_10_afc_lock_mode_offset);
   switch (ulTmp)
       {
       case radio_cfg_10_afc_lock_mode_free_running       :
           printf("\n   radio_cfg_10_afc_lock_mode_free_running       ");
           break;
       case radio_cfg_10_afc_lock_mode_disabled           :
           printf("\n   radio_cfg_10_afc_lock_mode_disabled           ");
           break;
       case radio_cfg_10_afc_lock_mode_paused             :
           printf("\n   radio_cfg_10_afc_lock_mode_paused             ");
           break;
       case radio_cfg_10_afc_lock_mode_lock_after_preamble:
           printf("\n   radio_cfg_10_afc_lock_mode_lock_after_preamble");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   printf("\n");

   printf("\nradio_cfg_11_r = 0x%02X",pBBRAM->radio_cfg_11_r);
   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_11_r,radio_cfg_11_afc_ki_2_numbits,radio_cfg_11_afc_ki_2_offset);
   switch (ulTmp)
       {
       case radio_cfg_11_afc_ki_2_power_0 :
           printf("\n   radio_cfg_11_afc_ki_2_power_0 ");
           break;
       case radio_cfg_11_afc_ki_2_power_1 :
           printf("\n   radio_cfg_11_afc_ki_2_power_1 ");
           break;
       case radio_cfg_11_afc_ki_2_power_2 :
           printf("\n   radio_cfg_11_afc_ki_2_power_2 ");
           break;
       case radio_cfg_11_afc_ki_2_power_3 :
           printf("\n   radio_cfg_11_afc_ki_2_power_3 ");
           break;
       case radio_cfg_11_afc_ki_2_power_4 :
           printf("\n   radio_cfg_11_afc_ki_2_power_4 ");
           break;
       case radio_cfg_11_afc_ki_2_power_5 :
           printf("\n   radio_cfg_11_afc_ki_2_power_5 ");
           break;
       case radio_cfg_11_afc_ki_2_power_6 :
           printf("\n   radio_cfg_11_afc_ki_2_power_6 ");
           break;
       case radio_cfg_11_afc_ki_2_power_7 :
           printf("\n   radio_cfg_11_afc_ki_2_power_7 ");
           break;
       case radio_cfg_11_afc_ki_2_power_8 :
           printf("\n   radio_cfg_11_afc_ki_2_power_8 ");
           break;
       case radio_cfg_11_afc_ki_2_power_9 :
           printf("\n   radio_cfg_11_afc_ki_2_power_9 ");
           break;
       case radio_cfg_11_afc_ki_2_power_10:
           printf("\n   radio_cfg_11_afc_ki_2_power_10");
           break;
       case radio_cfg_11_afc_ki_2_power_11:
           printf("\n   radio_cfg_11_afc_ki_2_power_11");
           break;
       case radio_cfg_11_afc_ki_2_power_12:
           printf("\n   radio_cfg_11_afc_ki_2_power_12");
           break;
       case radio_cfg_11_afc_ki_2_power_13:
           printf("\n   radio_cfg_11_afc_ki_2_power_13");
           break;
       case radio_cfg_11_afc_ki_2_power_14:
           printf("\n   radio_cfg_11_afc_ki_2_power_14");
           break;
       case radio_cfg_11_afc_ki_2_power_15:
           printf("\n   radio_cfg_11_afc_ki_2_power_15");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   ulTmp = XTRCT_BITS(pBBRAM->radio_cfg_11_r,radio_cfg_11_afc_kp_2_numbits,radio_cfg_11_afc_kp_2_offset);
   switch (ulTmp)
       {
       case radio_cfg_11_afc_kp_2_power_0 :
           printf("\n   radio_cfg_11_afc_kp_2_power_0 ");
           break;
       case radio_cfg_11_afc_kp_2_power_1 :
           printf("\n   radio_cfg_11_afc_kp_2_power_1 ");
           break;
       case radio_cfg_11_afc_kp_2_power_2 :
           printf("\n   radio_cfg_11_afc_kp_2_power_2 ");
           break;
       case radio_cfg_11_afc_kp_2_power_3 :
           printf("\n   radio_cfg_11_afc_kp_2_power_3 ");
           break;
       case radio_cfg_11_afc_kp_2_power_4 :
           printf("\n   radio_cfg_11_afc_kp_2_power_4 ");
           break;
       case radio_cfg_11_afc_kp_2_power_5 :
           printf("\n   radio_cfg_11_afc_kp_2_power_5 ");
           break;
       case radio_cfg_11_afc_kp_2_power_6 :
           printf("\n   radio_cfg_11_afc_kp_2_power_6 ");
           break;
       case radio_cfg_11_afc_kp_2_power_7 :
           printf("\n   radio_cfg_11_afc_kp_2_power_7 ");
           break;
       case radio_cfg_11_afc_kp_2_power_8 :
           printf("\n   radio_cfg_11_afc_kp_2_power_8 ");
           break;
       case radio_cfg_11_afc_kp_2_power_9 :
           printf("\n   radio_cfg_11_afc_kp_2_power_9 ");
           break;
       case radio_cfg_11_afc_kp_2_power_10:
           printf("\n   radio_cfg_11_afc_kp_2_power_10");
           break;
       case radio_cfg_11_afc_kp_2_power_11:
           printf("\n   radio_cfg_11_afc_kp_2_power_11");
           break;
       case radio_cfg_11_afc_kp_2_power_12:
           printf("\n   radio_cfg_11_afc_kp_2_power_12");
           break;
       case radio_cfg_11_afc_kp_2_power_13:
           printf("\n   radio_cfg_11_afc_kp_2_power_13");
           break;
       case radio_cfg_11_afc_kp_2_power_14:
           printf("\n   radio_cfg_11_afc_kp_2_power_14");
           break;
       case radio_cfg_11_afc_kp_2_power_15:
           printf("\n   radio_cfg_11_afc_kp_2_power_15");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   printf("\n");

   printf("\nimage_reject_cal_phase_r = 0x%02X",pBBRAM->image_reject_cal_phase_r);
   printf("\n");

   printf("\nimage_reject_cal_amplitude_r = 0x%02X",pBBRAM->image_reject_cal_amplitude_r);
   printf("\n");

   printf("\nmode_control_r = 0x%02X",pBBRAM->mode_control_r);
   if(pBBRAM->mode_control_r & mode_control_swm_en_enabled                              )
       printf("\n   mode_control_swm_en_enabled                             ");
   else
       printf("\n   mode_control_swm_en_disabled                            ");
   if(pBBRAM->mode_control_r & mode_control_bb_cal_enabled                              )
       printf("\n   mode_control_bb_cal_enabled                             ");
   else
       printf("\n   mode_control_bb_cal_disabled                            ");
   if(pBBRAM->mode_control_r & mode_control_swm_rssi_qual_enabled                       )
       printf("\n   mode_control_swm_rssi_qual_enabled                      ");
   else
       printf("\n   mode_control_swm_rssi_qual_disabled                     ");
   if(pBBRAM->mode_control_r & mode_control_tx_auto_turnaround_enabled                  )
       printf("\n   mode_control_tx_auto_turnaround_enabled                 ");
   else
       printf("\n   mode_control_tx_auto_turnaround_disabled                ");
   if(pBBRAM->mode_control_r & mode_control_rx_auto_turnaround_enabled                  )
       printf("\n   mode_control_rx_auto_turnaround_enabled                 ");
   else
       printf("\n   mode_control_rx_auto_turnaround_disabled                ");
   if(pBBRAM->mode_control_r & mode_control_custom_trx_synth_lock_time_en_enabled       )
       printf("\n   mode_control_custom_trx_synth_lock_time_en_enabled      ");
   else
       printf("\n   mode_control_custom_trx_synth_lock_time_en_disabled     ");
   if(pBBRAM->mode_control_r & mode_control_ext_lna_en_enabled                          )
       printf("\n   mode_control_ext_lna_en_enabled                         ");
   else
       printf("\n   mode_control_ext_lna_en_disabled                        ");
   if(pBBRAM->mode_control_r & mode_control_ext_pa_en_enabled                           )
       printf("\n   mode_control_ext_pa_en_enabled                          ");
   else
       printf("\n   mode_control_ext_pa_en_disabled                         ");

   printf("\n");

   printf("\npreamble_match_r = 0x%02X",pBBRAM->preamble_match_r);
   switch (pBBRAM->preamble_match_r)
       {
       case preamble_match_qual_disabled:
           printf("\n   preamble_match_qual_disabled");
           break;
       case preamble_match_4_in_24_win  :
           printf("\n   preamble_match_4_in_24_win  ");
           break;
       case preamble_match_3_in_24_win  :
           printf("\n   preamble_match_3_in_24_win  ");
           break;
       case preamble_match_2_in_24_win  :
           printf("\n   preamble_match_2_in_24_win  ");
           break;
       case preamble_match_1_in_24_win  :
           printf("\n   preamble_match_1_in_24_win  ");
           break;
       case preamble_match_0_in_24_win  :
           printf("\n   preamble_match_0_in_24_win  ");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   printf("\n");

   printf("\nsymbol_mode_r = 0x%02X",pBBRAM->symbol_mode_r);
   if(pBBRAM->symbol_mode_r & symbol_mode_symbol_length_10_bit         )
       printf("\n   symbol_mode_symbol_length_10_bit        ");
   else
       printf("\n   symbol_mode_symbol_length_8_bit         ");
   if(pBBRAM->symbol_mode_r & symbol_mode_data_whitening_enabled       )
       printf("\n   symbol_mode_data_whitening_enabled      ");
   else
       printf("\n   symbol_mode_data_whitening_disabled     ");
   if(pBBRAM->symbol_mode_r & symbol_mode_eight_ten_enc_enabled   )
       printf("\n   symbol_mode_eight_ten_enc_enabled  ");
   else
       printf("\n   symbol_mode_eight_ten_enc_disabled ");
   if(pBBRAM->symbol_mode_r & symbol_mode_prog_crc_en_enabled          )
       printf("\n   symbol_mode_prog_crc_en_enabled         ");
   else
       printf("\n   symbol_mode_prog_crc_en_disabled        ");
   if(pBBRAM->symbol_mode_r & symbol_mode_manchester_enc_enabled       )
       printf("\n   symbol_mode_manchester_enc_enabled      ");
   else
       printf("\n   symbol_mode_manchester_enc_disabled     ");



   printf("\n");

   printf("\npreamble_len_r (TX) = 0x%02X",pBBRAM->preamble_len_r);
   printf("\n   TX Preamble Len   %d Bits",pBBRAM->preamble_len_r * 8);
   printf("\n");

   printf("\ncrc_poly_0_r = 0x%02X",pBBRAM->crc_poly_0_r);
   printf("\ncrc_poly_1_r = 0x%02X",pBBRAM->crc_poly_1_r);
   ulTmp = MAKE_ULONG(pBBRAM->crc_poly_0_r,pBBRAM->crc_poly_1_r,0,0);
   printf("\n   CRC Poly   0x%04lX",ulTmp);
   printf("\n");

   printf("\nsync_control_r = 0x%02X",pBBRAM->sync_control_r);
   ulTmp = XTRCT_BITS(pBBRAM->sync_control_r,sync_control_sync_word_length_numbits,sync_control_sync_word_length_offset);
   switch (ulTmp)
       {
       case sync_control_sync_word_length_0 :
           printf("\n   sync_control_sync_word_length_0 ");
           break;
       case sync_control_sync_word_length_1 :
           printf("\n   sync_control_sync_word_length_1 ");
           break;
       case sync_control_sync_word_length_2 :
           printf("\n   sync_control_sync_word_length_2 ");
           break;
       case sync_control_sync_word_length_3 :
           printf("\n   sync_control_sync_word_length_3 ");
           break;
       case sync_control_sync_word_length_4 :
           printf("\n   sync_control_sync_word_length_4 ");
           break;
       case sync_control_sync_word_length_5 :
           printf("\n   sync_control_sync_word_length_5 ");
           break;
       case sync_control_sync_word_length_6 :
           printf("\n   sync_control_sync_word_length_6 ");
           break;
       case sync_control_sync_word_length_7 :
           printf("\n   sync_control_sync_word_length_7 ");
           break;
       case sync_control_sync_word_length_8 :
           printf("\n   sync_control_sync_word_length_8 ");
           break;
       case sync_control_sync_word_length_9 :
           printf("\n   sync_control_sync_word_length_9 ");
           break;
       case sync_control_sync_word_length_10:
           printf("\n   sync_control_sync_word_length_10");
           break;
       case sync_control_sync_word_length_11:
           printf("\n   sync_control_sync_word_length_11");
           break;
       case sync_control_sync_word_length_12:
           printf("\n   sync_control_sync_word_length_12");
           break;
       case sync_control_sync_word_length_13:
           printf("\n   sync_control_sync_word_length_13");
           break;
       case sync_control_sync_word_length_14:
           printf("\n   sync_control_sync_word_length_14");
           break;
       case sync_control_sync_word_length_15:
           printf("\n   sync_control_sync_word_length_15");
           break;
       case sync_control_sync_word_length_16:
           printf("\n   sync_control_sync_word_length_16");
           break;
       case sync_control_sync_word_length_17:
           printf("\n   sync_control_sync_word_length_17");
           break;
       case sync_control_sync_word_length_18:
           printf("\n   sync_control_sync_word_length_18");
           break;
       case sync_control_sync_word_length_19:
           printf("\n   sync_control_sync_word_length_19");
           break;
       case sync_control_sync_word_length_20:
           printf("\n   sync_control_sync_word_length_20");
           break;
       case sync_control_sync_word_length_21:
           printf("\n   sync_control_sync_word_length_21");
           break;
       case sync_control_sync_word_length_22:
           printf("\n   sync_control_sync_word_length_22");
           break;
       case sync_control_sync_word_length_23:
           printf("\n   sync_control_sync_word_length_23");
           break;
       case sync_control_sync_word_length_24:
           printf("\n   sync_control_sync_word_length_24");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   ulTmp = XTRCT_BITS(pBBRAM->sync_control_r,sync_control_sync_error_tol_numbits,sync_control_sync_error_tol_offset);
   switch (ulTmp)
       {
       case sync_control_sync_error_tol_0_errors_allowed :
           printf("\n   sync_control_sync_error_tol_0_errors_allowed ");
           break;
       case sync_control_sync_error_tol_1_errors_allowed    :
           printf("\n   sync_control_sync_error_tol_1_errors_allowed    ");
           break;
       case sync_control_sync_error_tol_2_errors_allowed    :
           printf("\n   sync_control_sync_error_tol_2_errors_allowed    ");
           break;
       case sync_control_sync_error_tol_3_errors_allowed    :
           printf("\n   sync_control_sync_error_tol_3_errors_allowed    ");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   printf("\n");

   printf("\nsync_byte_0_r = 0x%02X",pBBRAM->sync_byte_0_r);
   printf("\nsync_byte_1_r = 0x%02X",pBBRAM->sync_byte_1_r);
   printf("\nsync_byte_2_r = 0x%02X",pBBRAM->sync_byte_2_r);
   ulTmp = XTRCT_BITS(pBBRAM->sync_control_r,sync_control_sync_word_length_numbits,sync_control_sync_word_length_offset);
   switch (ulTmp)
       {
       case sync_control_sync_word_length_0 :
           printf("\n   No Sync transmitted ");
           break;
       case sync_control_sync_word_length_1 :
       case sync_control_sync_word_length_2 :
       case sync_control_sync_word_length_3 :
       case sync_control_sync_word_length_4 :
       case sync_control_sync_word_length_5 :
       case sync_control_sync_word_length_6 :
       case sync_control_sync_word_length_7 :
       case sync_control_sync_word_length_8 :
           printf("\nsync_byte_2_r transmitted");
           break;
       case sync_control_sync_word_length_9 :
       case sync_control_sync_word_length_10:
       case sync_control_sync_word_length_11:
       case sync_control_sync_word_length_12:
       case sync_control_sync_word_length_13:
       case sync_control_sync_word_length_14:
       case sync_control_sync_word_length_15:
       case sync_control_sync_word_length_16:
           printf("\nsync_byte_1_r + sync_byte_2_r transmitted");
           break;
       case sync_control_sync_word_length_17:
       case sync_control_sync_word_length_18:
       case sync_control_sync_word_length_19:
       case sync_control_sync_word_length_20:
       case sync_control_sync_word_length_21:
       case sync_control_sync_word_length_22:
       case sync_control_sync_word_length_23:
       case sync_control_sync_word_length_24:
           printf("\nsync_byte_0_r + sync_byte_1_r + sync_byte_2_r transmitted");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }

   printf("\n");

   printf("\ntx_base_adr_r = 0x%02X",pBBRAM->tx_base_adr_r);
   printf("\nrx_base_adr_r = 0x%02X",pBBRAM->rx_base_adr_r);
   printf("\n");

   printf("\npacket_length_control_r = 0x%02X",pBBRAM->packet_length_control_r);
   if(pBBRAM->packet_length_control_r & packet_length_control_data_byte_msb)
       printf("\n   packet_length_control_data_byte_msb ");
   else
       printf("\n   packet_length_control_data_byte_lsb ");
   if(pBBRAM->packet_length_control_r & packet_length_control_packet_len_fixed)
       printf("\n   packet_length_control_packet_len_fixed");
   else
       printf("\n   packet_length_control_packet_len_variable");

   if(pBBRAM->packet_length_control_r & packet_length_control_crc_en_yes)
       printf("\n   packet_length_control_crc_en_yes");
   else
       printf("\n   packet_length_control_crc_en_no");

   ulTmp = XTRCT_BITS(pBBRAM->packet_length_control_r,packet_length_control_length_offset_numbits,packet_length_control_length_offset_offset);
   switch (ulTmp)
       {
       case packet_length_control_length_offset_minus4:
           printf("\n   packet_length_control_length_offset_minus4");
           break;
       case packet_length_control_length_offset_minus3:
           printf("\n   packet_length_control_length_offset_minus3");
           break;
       case packet_length_control_length_offset_minus2:
           printf("\n   packet_length_control_length_offset_minus2");
           break;
       case packet_length_control_length_offset_minus1:
           printf("\n   packet_length_control_length_offset_minus1");
           break;
       case packet_length_control_length_offset_minus0:
           printf("\n   packet_length_control_length_offset_minus0");
           break;
       case packet_length_control_length_offset_plus1 :
           printf("\n   packet_length_control_length_offset_plus1 ");
           break;
       case packet_length_control_length_offset_plus2 :
           printf("\n   packet_length_control_length_offset_plus2 ");
           break;
       case packet_length_control_length_offset_plus3 :
           printf("\n   packet_length_control_length_offset_plus3 ");
           break;
       default:
           printf("\n   !!INVALID!!  ");
           break;
       }
   ulTmp = XTRCT_BITS(pBBRAM->packet_length_control_r,packet_length_control_data_mode_numbits,packet_length_control_data_mode_offset);
   switch (ulTmp)
       {
       case packet_length_control_data_mode_packet      :
           printf("\n   packet_length_control_data_mode_packet      ");
           break;
       case packet_length_control_data_mode_sport_preamble:
           printf("\n   packet_length_control_data_mode_sport_preamble");
           break;
       case packet_length_control_data_mode_sport_sync    :
           printf("\n   packet_length_control_data_mode_sport_sync    ");
           break;
       case packet_length_control_data_mode_unused        :
           printf("\n   packet_length_control_data_mode_unused        ");
           break;
       default:
           printf("\n   !!INVALID SPORT!!  ");
           break;
       }
   printf("\n");

   printf("\npacket_length_max_r = 0x%02X",pBBRAM->packet_length_max_r);
   printf("\n");

   printf("\nStatic_Reg_fix_r = 0x%02X",pBBRAM->static_reg_fix_r);
   printf("\n");


   printf("\nAddress_Match_Offset_r = 0x%02X",pBBRAM->address_match_offset_r);
   printf("\n");

   for (iOffset = 0x0; iOffset < sizeof(pBBRAM->address_filtering_r);iOffset++)
       printf("\naddress_filtering_r[%d] = 0x%02X",iOffset,pBBRAM->address_filtering_r[iOffset]);

   printf("\nrx_synth_lock_time_r = 0x%02X",pBBRAM->rx_synth_lock_time_r);
   printf("\nrx_synth_lock_time_r = 0x%02X",pBBRAM->rx_synth_lock_time_r);

   printf("\n");

   // Dump in Raw mode also so that we can compare in Excel
   pTmp  = (unsigned char *)pBBRAM;

   for (iCnt = 0; iCnt < sizeof(TyBBRAM);iCnt++)
      printf("\n%02X",pTmp[iCnt]);
}   




